function [ang,ang1,isGroove] = GetGrooves(r1,r2,x_spiral,y_spiral,n_grooves,beta,delta_beta,R,Theta)
% FUNCTION TO IDENTIFY THE LOCATIONS OF LOGARITHMIC SPIRAL GROOVES IN A 2D CIRCULAR MESH  
%  
% Inputs:
% GEOMETRY DEFINED ACCORDING TO MUIJDERMAN NOTATION:
% r1 [m] - Innermost radius of the bearing  
% r2 [m] - Outer radius of the bearing  
% x_spiral [adim] - Ratio of groove widths (a1/a2)  
% y_spiral [adim] - Normalized groove position, defined as (r2-rb)/(r2-r1)  
% n_grooves - Number of spiral grooves  
% beta [°] - Base angle of the logarithmic spiral  
% delta_beta [°] - Phase shift of the spiral with respect to the 0° reference line  
% R [m] - Array of radial coordinates (MxN)  
% Theta [°] - Array of angular coordinates (MxN)  

% Outputs:
% ang - Array containing the angular position of the logarithmic spiral groove at different radii  
% ang1 - Array containing the angular position of the second boundary of the groove  
% isGroove - Logical array of the same size as the input mesh, where 'true'
% (1) indicates a groove region and 'false' (0) indicates a land region  

isGroove = false(size(R)); % Initialize logical array for groove locations  
Theta = Theta - pi; % Shift the angular domain to be within [-pi, pi] for consistency  

%% LOGARITHMIC GROOVE GEOMETRY  
passo = 2.*pi.*r2./n_grooves;   % Circumferential pitch (spacing) of grooves  
delta_theta = passo./r2;        % Angular separation between grooves  
a2 = passo./(1+x_spiral);       % Groove width (a2) based on given ratio  
a1 = passo-a2;                  % Complementary land width (a1)  
b = y_spiral.*(r2-r1);          % Radial position of the base of the groove  
rb = r2-b;                      % Base radius of the grooves (starting point of the spiral)  
beta = beta/180*pi;             % Convert beta from degrees to radians  

% Generate logarithmic spiral curves for the groove boundaries  
r = linspace(rb,r2,100);        % Radial discretization for the spiral  
angle = -tan(beta).*log(r./r2) - delta_beta; % Logarithmic spiral equation for first boundary  
angle1 = angle + a1./r2;        % Second boundary of the groove (phase-shifted)  

% Define radial points where grooves start within the mesh  
R_logic = R(1,R(1,:)>=rb);      % Extract radial coordinates where r >= rb  
[~,pos] = min(abs(R(1,:)-rb));  % Find column index where r = rb  

% Compute angular positions of groove boundaries for these radial points  
angle_logic = -tan(beta).*log(R_logic./r2) - delta_beta;  
angle1_logic = angle_logic + a1/r2;  

%% GROOVE IDENTIFICATION  
% Initialize arrays to store angular values for each groove  
ang = zeros(length(angle),n_grooves);  
ang1 = zeros(length(angle1),n_grooves);  

% Adjust boundary angles for each groove iteration  
angle_logic = angle_logic - delta_theta;  
angle1_logic = angle1_logic - delta_theta;  

for i = 1:n_grooves  
    % Define angular positions for the current groove  
    ang(:,i) = angle + (i-1) * delta_theta;  
    ang1(:,i) = angle1 + (i-1) * delta_theta;  
    angle_logic = angle_logic + delta_theta;  
    angle1_logic = angle1_logic + delta_theta;  

    % Ensure angular values remain within the [-pi, pi] domain  
    angle_logic(angle_logic>pi) = angle_logic(angle_logic>pi) - 2*pi;  
    angle1_logic(angle1_logic>pi) = angle1_logic(angle1_logic>pi) - 2*pi;  

    % Identify cases where the groove transitions across the -pi to pi boundary  
    pos_log = angle_logic > 0 & angle1_logic < 0;  
    if ~isempty(find(pos_log, 1))  
        % If a groove crosses the angular boundary, handle it separately  
        pos_log = find(pos_log);  
        Theta_log = [false(size(Theta,1),pos-1), ...
                     Theta(:,pos:end) > angle_logic & Theta(:,pos:end) < angle1_logic];  

        Theta_log = Theta_log | [false(size(Theta,1),pos+pos_log(1)-2), ...
                     Theta(:,pos_log) > angle_logic(pos_log) | Theta(:,pos_log) < angle1_logic(pos_log), ...
                     false(size(Theta,1),size(Theta,2)-pos_log(end)-pos+1)];  

    else  
        % Standard case where groove does not cross the angular boundary  
        Theta_log = [false(size(Theta,1),pos-1), ...
                     Theta(:,pos:end) > angle_logic & Theta(:,pos:end) < angle1_logic];  
    end  

    % Update the global logical array, ensuring that only regions inside the groove are marked  
    isGroove = isGroove | Theta_log;  
    isGroove = isGroove & (R >= rb);  
end  

end  
